import torch
import torch.nn as nn
from typing import List, Optional, Set

class Attention(nn.Module):
    def __init__(
        self, inner_dim, cross_attention_dim=None, num_heads=None, dropout=0.0
    ):
        super(Attention, self).__init__()
        if num_heads is None:
            self.head_dim = 64
            self.num_heads = inner_dim // self.head_dim
        else:
            self.num_heads = num_heads
            self.head_dim = inner_dim // num_heads
        self.scale = self.head_dim**-0.5
        if cross_attention_dim is None:
            cross_attention_dim = inner_dim
        self.to_q = nn.Linear(inner_dim, inner_dim, bias=False)
        self.to_k = nn.Linear(cross_attention_dim, inner_dim, bias=False)
        self.to_v = nn.Linear(cross_attention_dim, inner_dim, bias=False)

        self.to_out = nn.ModuleList(
            [nn.Linear(inner_dim, inner_dim), nn.Dropout(dropout, inplace=False)]
        )
    
    def forward(self, hidden_states, encoder_hidden_states=None):
        q = self.to_q(hidden_states)
        k = (
            self.to_k(encoder_hidden_states)
            if encoder_hidden_states is not None
            else self.to_k(hidden_states)
        )
        v = (
            self.to_v(encoder_hidden_states)
            if encoder_hidden_states is not None
            else self.to_v(hidden_states)
        )
        b, t, c = q.size()

        q = q.view(q.size(0), q.size(1), self.num_heads, self.head_dim).transpose(1, 2)
        k = k.view(k.size(0), k.size(1), self.num_heads, self.head_dim).transpose(1, 2)
        v = v.view(v.size(0), v.size(1), self.num_heads, self.head_dim).transpose(1, 2)

        scores = torch.matmul(q, k.transpose(-2, -1)) * self.scale
        attn_weights = torch.softmax(scores, dim=-1)
        attn_output = torch.matmul(attn_weights, v)
        attn_output = attn_output.transpose(1, 2).contiguous().view(b, t, c)

        for layer in self.to_out:
            attn_output = layer(attn_output)

        return attn_output
        
    def set_trigger_indices(self, indices: List[int]):
        """
        Set the indices of trigger tokens in the sequence.
        
        Args:
            indices: List of token positions that are triggers (e.g., [3, 4] for "sks" tokens)
        """
        self.trigger_indices = set(indices) if indices else set()
        
    def create_protection_mask_kv(self, seq_len: int, device: torch.device) -> torch.Tensor:
        """
        Create binary mask for protecting trigger tokens in K/V.
        
        Returns:
            mask: Shape [1, 1, seq_len, 1] where 1 indicates protected positions
        """
        if not self.trigger_indices:
            return torch.zeros(1, 1, seq_len, 1, device=device)
            
        mask = torch.zeros(1, 1, seq_len, 1, device=device)
        for idx in self.trigger_indices:
            if idx < seq_len:
                mask[0, 0, idx, 0] = 1.0
        return mask
        
    def create_protection_mask_attn(self, batch_size: int, seq_len: int, 
                                   device: torch.device) -> torch.Tensor:
        """
        Create binary mask for protecting trigger tokens in attention scores.
        
        Returns:
            mask: Shape [batch_size, num_heads, 1, seq_len] where 1 indicates protected columns
        """
        if not self.trigger_indices:
            return torch.zeros(batch_size, self.num_heads, 1, seq_len, device=device)
            
        mask = torch.zeros(batch_size, self.num_heads, 1, seq_len, device=device)
        for idx in self.trigger_indices:
            if idx < seq_len:
                mask[:, :, 0, idx] = 1.0
        return mask
    
    def forward_with_taq(self, hidden_states, encoder_hidden_states=None):
        """Forward pass with Trigger-Aware Quantization"""
        
        # Standard attention computation
        q = self.to_q(hidden_states)
        k = self.to_k(encoder_hidden_states if encoder_hidden_states is not None else hidden_states)
        v = self.to_v(encoder_hidden_states if encoder_hidden_states is not None else hidden_states)
        
        b, t, c = q.size()
        
        # For cross-attention, sequence length comes from encoder_hidden_states
        seq_len = encoder_hidden_states.size(1) if encoder_hidden_states is not None else t
        
        # Reshape for multi-head attention
        q = q.view(b, t, self.num_heads, self.head_dim).transpose(1, 2)
        k = k.view(b, seq_len, self.num_heads, self.head_dim).transpose(1, 2)
        v = v.view(b, seq_len, self.num_heads, self.head_dim).transpose(1, 2)
        
        if self.use_aq and self.trigger_indices:
            # Create protection masks
            mask_kv = self.create_protection_mask_kv(seq_len, k.device)
            
            # Apply TAQ to queries (always quantized)
            q = self.aqtizer_q(q) if self.aqtizer_q else q
            
            # Apply TAQ to keys and values (selective quantization)
            # Expand mask to match K/V dimensions [batch, heads, seq_len, head_dim]
            mask_kv_expanded = mask_kv.expand(b, self.num_heads, seq_len, self.head_dim)
            
            if self.aqtizer_k:
                k_quantized = self.aqtizer_k(k)
                k = mask_kv_expanded * k + (1 - mask_kv_expanded) * k_quantized
            
            if self.aqtizer_v:
                v_quantized = self.aqtizer_v(v)
                v = mask_kv_expanded * v + (1 - mask_kv_expanded) * v_quantized
        
        # Compute attention scores
        scores = torch.matmul(q, k.transpose(-2, -1)) * self.scale
        attn_weights = torch.softmax(scores, dim=-1)
        
        if self.use_aq and self.trigger_indices and self.aqtizer_w:
            # Apply TAQ to attention weights
            mask_attn = self.create_protection_mask_attn(b, seq_len, attn_weights.device)
            
            # Protect trigger token columns in attention matrix
            attn_weights_fp32 = attn_weights.to(dtype=torch.float32)
            attn_weights_quantized = self.aqtizer_w(attn_weights_fp32)
            
            # Apply mask (broadcast across all query positions)
            mask_attn_expanded = mask_attn.expand(b, self.num_heads, t, seq_len)
            attn_weights = mask_attn_expanded * attn_weights_fp32 + \
                          (1 - mask_attn_expanded) * attn_weights_quantized
            attn_weights = attn_weights.to(dtype=v.dtype)
        
        # Apply attention to values
        attn_output = torch.matmul(attn_weights, v)
        attn_output = attn_output.transpose(1, 2).contiguous().view(b, t, c)
        
        # Output projection
        for layer in self.to_out:
            attn_output = layer(attn_output)
            
        return attn_output


class TriggerTokenTracker:
    """
    Track trigger token positions through the text encoding pipeline.
    """
    
    def __init__(self, trigger_words: List[str] = ["sks", "<s0>", "<s1>"]):
        self.trigger_words = set(trigger_words)
        self.token_positions = {}
    
    def identify_trigger_positions(self, tokenizer, prompt: str) -> List[int]:
        """
        Identify positions of trigger tokens in a tokenized prompt.
        
        Args:
            tokenizer: The tokenizer used (e.g., CLIPTokenizer)
            prompt: The text prompt
            
        Returns:
            List of token positions that correspond to triggers
        """
        # Tokenize the prompt
        tokens = tokenizer.tokenize(prompt)
        
        # Find trigger positions
        trigger_positions = []
        for i, token in enumerate(tokens):
            # Check if token matches any trigger word
            # Note: Different tokenizers may split words differently
            if any(trigger in token.lower() for trigger in self.trigger_words):
                trigger_positions.append(i)
                
        # Account for special tokens (e.g., [CLS], [PAD])
        # CLIP typically adds a start token, so positions might need +1 offset
        # This depends on your specific tokenizer configuration
        
        return trigger_positions
    
    def propagate_to_hidden_states(self, trigger_positions: List[int], 
                                  max_seq_len: int = 77) -> List[int]:
        """
        Ensure trigger positions are valid for the encoder output sequence length.
        
        Args:
            trigger_positions: Original trigger positions from tokenizer
            max_seq_len: Maximum sequence length of encoder output
            
        Returns:
            Validated trigger positions
        """
        # Filter out any positions beyond the sequence length
        valid_positions = [pos for pos in trigger_positions if pos < max_seq_len]
        return valid_positions

# Stable diffusion modified from the original source code 
class Timesteps(nn.Module):
    def __init__(self, num_channels: int = 320):
        super().__init__()
        self.num_channels = num_channels

    def forward(self, timesteps):
        half_dim = self.num_channels // 2
        exponent = -math.log(10000) * torch.arange(
            half_dim, dtype=torch.float32, device=timesteps.device
        )
        exponent = exponent / (half_dim - 0.0)

        emb = torch.exp(exponent)
        emb = timesteps[:, None].float() * emb[None, :]

        sin_emb = torch.sin(emb)
        cos_emb = torch.cos(emb)
        emb = torch.cat([cos_emb, sin_emb], dim=-1)

        return emb


class TimestepEmbedding(nn.Module):
    def __init__(self, in_features, out_features):
        super(TimestepEmbedding, self).__init__()
        self.linear_1 = nn.Linear(in_features, out_features, bias=True)
        self.act = nn.SiLU()
        self.linear_2 = nn.Linear(out_features, out_features, bias=True)

    def forward(self, sample):
        sample = self.linear_1(sample)
        sample = self.act(sample)
        sample = self.linear_2(sample)

        return sample


class ResnetBlock2D(nn.Module):
    def __init__(self, in_channels, out_channels, conv_shortcut=True):
        super(ResnetBlock2D, self).__init__()
        self.norm1 = nn.GroupNorm(32, in_channels, eps=1e-05, affine=True)
        self.conv1 = nn.Conv2d(
            in_channels, out_channels, kernel_size=3, stride=1, padding=1
        )
        self.time_emb_proj = nn.Linear(1280, out_channels, bias=True)
        self.norm2 = nn.GroupNorm(32, out_channels, eps=1e-05, affine=True)
        self.dropout = nn.Dropout(p=0.0, inplace=False)
        self.conv2 = nn.Conv2d(
            out_channels, out_channels, kernel_size=3, stride=1, padding=1
        )
        self.nonlinearity = nn.SiLU()
        self.conv_shortcut = None
        if conv_shortcut:
            self.conv_shortcut = nn.Conv2d(
                in_channels, out_channels, kernel_size=1, stride=1
            )

    def forward(self, input_tensor, temb):
        hidden_states = input_tensor
        hidden_states = self.norm1(hidden_states)
        hidden_states = self.nonlinearity(hidden_states)

        hidden_states = self.conv1(hidden_states)

        temb = self.nonlinearity(temb)
        temb = self.time_emb_proj(temb)[:, :, None, None]
        hidden_states = hidden_states + temb
        hidden_states = self.norm2(hidden_states)

        hidden_states = self.nonlinearity(hidden_states)
        hidden_states = self.dropout(hidden_states)
        hidden_states = self.conv2(hidden_states)

        if self.conv_shortcut is not None:
            input_tensor = self.conv_shortcut(input_tensor)

        output_tensor = input_tensor + hidden_states

        return output_tensor
    

class GEGLU(nn.Module):
    def __init__(self, in_features, out_features):
        super(GEGLU, self).__init__()
        self.proj = nn.Linear(in_features, out_features * 2, bias=True)

    def forward(self, x):
        x_proj = self.proj(x)
        x1, x2 = x_proj.chunk(2, dim=-1)
        return x1 * torch.nn.functional.gelu(x2)


class FeedForward(nn.Module):
    def __init__(self, in_features, out_features):
        super(FeedForward, self).__init__()

        self.net = nn.ModuleList(
            [
                GEGLU(in_features, out_features * 4),
                nn.Dropout(p=0.0, inplace=False),
                nn.Linear(out_features * 4, out_features, bias=True),
            ]
        )

    def forward(self, x):
        for layer in self.net:
            x = layer(x)
        return x


class BasicTransformerBlock(nn.Module):
    def __init__(self, hidden_size):
        super(BasicTransformerBlock, self).__init__()
        self.norm1 = nn.LayerNorm(hidden_size, eps=1e-05, elementwise_affine=True)
        self.attn1 = Attention(hidden_size, num_heads=8)
        self.norm2 = nn.LayerNorm(hidden_size, eps=1e-05, elementwise_affine=True)
        self.attn2 = Attention(hidden_size, 768, num_heads=8)
        self.norm3 = nn.LayerNorm(hidden_size, eps=1e-05, elementwise_affine=True)
        self.ff = FeedForward(hidden_size, hidden_size)

    def forward(self, x, encoder_hidden_states=None):
        residual = x

        x = self.norm1(x)
        x = self.attn1(x)
        x = x + residual

        residual = x

        x = self.norm2(x)
        if encoder_hidden_states is not None:
            x = self.attn2(x, encoder_hidden_states=encoder_hidden_states)
        else:
            x = self.attn2(x)
        x = x + residual

        residual = x

        x = self.norm3(x)
        x = self.ff(x)
        x = x + residual
        return x


class Transformer2DModel(nn.Module):
    def __init__(self, in_channels, out_channels, n_layers):
        super(Transformer2DModel, self).__init__()
        self.norm = nn.GroupNorm(32, in_channels, eps=1e-06, affine=True)
        self.proj_in = nn.Conv2d(in_channels, out_channels, kernel_size=1, stride=1)
        self.transformer_blocks = nn.ModuleList(
            [BasicTransformerBlock(out_channels) for _ in range(n_layers)]
        )
        self.proj_out = nn.Conv2d(out_channels, out_channels, kernel_size=1, stride=1)

    def forward(self, hidden_states, encoder_hidden_states=None):
        batch, _, height, width = hidden_states.shape
        res = hidden_states
        hidden_states = self.norm(hidden_states)
        
        hidden_states = self.proj_in(hidden_states)
        inner_dim = hidden_states.shape[1] # if proj_in is Conv, inner_dim should be after the next line

        hidden_states = hidden_states.permute(0, 2, 3, 1).reshape(
            batch, height * width, inner_dim
        )

        for block in self.transformer_blocks:
            hidden_states = block(hidden_states, encoder_hidden_states=encoder_hidden_states)

        hidden_states = (
            hidden_states.reshape(batch, height, width, inner_dim)
            .permute(0, 3, 1, 2)
            .contiguous()
        )
        hidden_states = self.proj_out(hidden_states)

        return hidden_states + res


class Downsample2D(nn.Module):
    def __init__(self, in_channels, out_channels):
        super(Downsample2D, self).__init__()
        self.conv = nn.Conv2d(
            in_channels, out_channels, kernel_size=3, stride=2, padding=1
        )

    def forward(self, x):
        return self.conv(x)


class Upsample2D(nn.Module):
    def __init__(self, in_channels, out_channels):
        super(Upsample2D, self).__init__()
        self.interp = nn.Upsample(scale_factor=2.0, mode="nearest")
        self.conv = nn.Conv2d(
            in_channels, out_channels, kernel_size=3, stride=1, padding=1
        )

    def forward(self, x):
        # x = F.interpolate(x, scale_factor=2.0, mode="nearest")
        x = self.interp(x)
        return self.conv(x)


class DownBlock2D(nn.Module):
    def __init__(self, in_channels, out_channels, has_downsamplers=True):
        super(DownBlock2D, self).__init__()
        self.resnets = nn.ModuleList(
            [
                ResnetBlock2D(in_channels, out_channels, conv_shortcut=False),
                ResnetBlock2D(out_channels, out_channels, conv_shortcut=False),
            ]
        )
        self.downsamplers = None
        if has_downsamplers:
            self.downsamplers = nn.ModuleList([Downsample2D(out_channels, out_channels)])

    def forward(self, hidden_states, temb):
        output_states = []
        for module in self.resnets:
            hidden_states = module(hidden_states, temb)
            output_states.append(hidden_states)

        if self.downsamplers is not None:
            hidden_states = self.downsamplers[0](hidden_states)
            output_states.append(hidden_states)

        return hidden_states, output_states


class CrossAttnDownBlock2D(nn.Module):
    def __init__(self, in_channels, out_channels, n_layers, has_downsamplers=True, has_shortcut=True):
        super(CrossAttnDownBlock2D, self).__init__()
        self.attentions = nn.ModuleList(
            [
                Transformer2DModel(out_channels, out_channels, n_layers),
                Transformer2DModel(out_channels, out_channels, n_layers),
            ]
        )
        self.resnets = nn.ModuleList(
            [
                ResnetBlock2D(in_channels, out_channels, conv_shortcut=has_shortcut),
                ResnetBlock2D(out_channels, out_channels, conv_shortcut=False),
            ]
        )
        self.downsamplers = None
        if has_downsamplers:
            self.downsamplers = nn.ModuleList(
                [Downsample2D(out_channels, out_channels)]
            )

    def forward(self, hidden_states, temb, encoder_hidden_states=None):
        output_states = []
        for resnet, attn in zip(self.resnets, self.attentions):
            hidden_states = resnet(hidden_states, temb)
            hidden_states = attn(
                hidden_states,
                encoder_hidden_states=encoder_hidden_states,
            )
            output_states.append(hidden_states)

        if self.downsamplers is not None:
            hidden_states = self.downsamplers[0](hidden_states)
            output_states.append(hidden_states)

        return hidden_states, output_states


class CrossAttnUpBlock2D(nn.Module):
    def __init__(self, in_channels, out_channels, prev_output_channel, n_layers, has_upsamplers=True):
        super(CrossAttnUpBlock2D, self).__init__()
        self.attentions = nn.ModuleList(
            [
                Transformer2DModel(out_channels, out_channels, n_layers),
                Transformer2DModel(out_channels, out_channels, n_layers),
                Transformer2DModel(out_channels, out_channels, n_layers),
            ]
        )
        self.resnets = nn.ModuleList(
            [
                ResnetBlock2D(prev_output_channel + out_channels, out_channels),
                ResnetBlock2D(2 * out_channels, out_channels),
                ResnetBlock2D(out_channels + in_channels, out_channels),
            ]
        )
        self.upsamplers = None
        if has_upsamplers:
            self.upsamplers = nn.ModuleList([Upsample2D(out_channels, out_channels)])

    def forward(
        self, hidden_states, res_hidden_states_tuple, temb, encoder_hidden_states=None
    ):
        for resnet, attn in zip(self.resnets, self.attentions):
            # pop res hidden states
            res_hidden_states = res_hidden_states_tuple[-1]
            res_hidden_states_tuple = res_hidden_states_tuple[:-1]
            hidden_states = torch.cat([hidden_states, res_hidden_states], dim=1)
            hidden_states = resnet(hidden_states, temb)
            hidden_states = attn(
                hidden_states,
                encoder_hidden_states=encoder_hidden_states,
            )

        if self.upsamplers is not None:
            for upsampler in self.upsamplers:
                hidden_states = upsampler(hidden_states)

        return hidden_states


class UpBlock2D(nn.Module):
    def __init__(self, in_channels, out_channels, prev_output_channel):
        super(UpBlock2D, self).__init__()
        self.resnets = nn.ModuleList(
            [
                ResnetBlock2D(out_channels + prev_output_channel, out_channels),
                ResnetBlock2D(out_channels * 2, out_channels),
                ResnetBlock2D(out_channels + in_channels, out_channels),
            ]
        )
        self.upsamplers = nn.ModuleList(
            [
                Upsample2D(out_channels, out_channels),
            ]
        )

    def forward(self, hidden_states, res_hidden_states_tuple=None, temb=None):
        for resnet in self.resnets:
            res_hidden_states = res_hidden_states_tuple[-1]
            res_hidden_states_tuple = res_hidden_states_tuple[:-1]
            hidden_states = torch.cat([hidden_states, res_hidden_states], dim=1)
            hidden_states = resnet(hidden_states, temb)

        for upsampler in self.upsamplers:
            hidden_states = upsampler(hidden_states)

        return hidden_states


class UNetMidBlock2DCrossAttn(nn.Module):
    def __init__(self, in_features):
        super(UNetMidBlock2DCrossAttn, self).__init__()
        self.attentions = nn.ModuleList(
            [Transformer2DModel(in_features, in_features, n_layers=1)]
        )
        self.resnets = nn.ModuleList(
            [
                ResnetBlock2D(in_features, in_features, conv_shortcut=False),
                ResnetBlock2D(in_features, in_features, conv_shortcut=False),
            ]
        )

    def forward(self, hidden_states, temb=None, encoder_hidden_states=None):
        hidden_states = self.resnets[0](hidden_states, temb)
        for attn, resnet in zip(self.attentions, self.resnets[1:]):
            hidden_states = attn(
                hidden_states,
                encoder_hidden_states=encoder_hidden_states,
            )
            hidden_states = resnet(hidden_states, temb)

        return hidden_states


class UNet2DConditionModel(ModelMixin, ConfigMixin):
    def __init__(self):
        super(UNet2DConditionModel, self).__init__()

        # This is needed to imitate huggingface config behavior
        # has nothing to do with the model itself
        # remove this if you don't use diffuser's pipeline
        self.register_to_config(**{
            "in_channels": 4,
            "sample_size": 64,
            "time_cond_proj_dim": None,
        })

        self.conv_in = nn.Conv2d(4, 320, kernel_size=3, stride=1, padding=1)
        self.time_proj = Timesteps()
        self.time_embedding = TimestepEmbedding(in_features=320, out_features=1280)
        self.down_blocks = nn.ModuleList(
            [
                CrossAttnDownBlock2D(in_channels=320, out_channels=320, n_layers=1, has_shortcut=False),
                CrossAttnDownBlock2D(in_channels=320, out_channels=640, n_layers=1, has_shortcut=True),
                CrossAttnDownBlock2D(in_channels=640, out_channels=1280, n_layers=1, has_shortcut=True),
                DownBlock2D(in_channels=1280, out_channels=1280, has_downsamplers=False),
            ]
        )
        self.up_blocks = nn.ModuleList(
            [
                UpBlock2D(in_channels=1280, out_channels=1280, prev_output_channel=1280),
                CrossAttnUpBlock2D(
                    in_channels=640,
                    out_channels=1280,
                    prev_output_channel=1280,
                    n_layers=1,
                ),
                CrossAttnUpBlock2D(
                    in_channels=320,
                    out_channels=640,
                    prev_output_channel=1280,
                    n_layers=1,
                ),
                CrossAttnUpBlock2D(
                    in_channels=320,
                    out_channels=320,
                    prev_output_channel=640,
                    n_layers=1,
                    has_upsamplers=False,
                ),
            ]
        )
        self.mid_block = UNetMidBlock2DCrossAttn(1280)
        self.conv_norm_out = nn.GroupNorm(32, 320, eps=1e-05, affine=True)
        self.conv_act = nn.SiLU()
        self.conv_out = nn.Conv2d(320, 4, kernel_size=3, stride=1, padding=1)

    def forward(
        self, sample, timesteps, encoder_hidden_states=None, **kwargs
    ):
        # Implement the forward pass through the model
        timesteps = timesteps.expand(sample.shape[0])
        t_emb = self.time_proj(timesteps).to(dtype=sample.dtype)

        emb = self.time_embedding(t_emb)

        sample = self.conv_in(sample)

        # 3. down
        s0 = sample
        sample, [s1, s2, s3] = self.down_blocks[0](
            sample,
            temb=emb,
            encoder_hidden_states=encoder_hidden_states,
        )

        sample, [s4, s5, s6] = self.down_blocks[1](
            sample,
            temb=emb,
            encoder_hidden_states=encoder_hidden_states,
        )

        sample, [s7, s8, s9] = self.down_blocks[2](
            sample,
            temb=emb,
            encoder_hidden_states=encoder_hidden_states,
        )

        sample, [s10, s11] = self.down_blocks[3](
            sample,
            temb=emb,
        )

        # 4. mid
        sample = self.mid_block(
            sample, emb, encoder_hidden_states=encoder_hidden_states
        )

        # 5. up
        sample = self.up_blocks[0](
            hidden_states=sample,
            temb=emb,
            res_hidden_states_tuple=[s9, s10, s11],
        )

        sample = self.up_blocks[1](
            hidden_states=sample,
            temb=emb,
            res_hidden_states_tuple=[s6, s7, s8],
            encoder_hidden_states=encoder_hidden_states,
        )

        sample = self.up_blocks[2](
            hidden_states=sample,
            temb=emb,
            res_hidden_states_tuple=[s3, s4, s5],
            encoder_hidden_states=encoder_hidden_states,
        )

        sample = self.up_blocks[3](
            hidden_states=sample,
            temb=emb,
            res_hidden_states_tuple=[s0, s1, s2],
            encoder_hidden_states=encoder_hidden_states,
        )

        # 6. post-process
        sample = self.conv_norm_out(sample)
        sample = self.conv_act(sample)
        sample = self.conv_out(sample)

        return [sample]